/************************************************************************
 * @file: gst_dlt_adapter.h
 *
 * @version: 0.1
 *
 * @description: This header file contains all the typedef, enum, struct
 * and API required to use GDA component.
 *
 * @authors: Athreya Manasa Balaraj, ManasaBalaraj.Athreya@in.bosch.com 2017
 *           Devi Laxmi, Laxmi.Devi@in.bosch.com 2017
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef __log_gst_dlt_H__
#define __log_gst_dlt_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include <dlt/dlt.h>
#include <gst/gst.h>

/**
 * @brief Macro for gda_register_logging function.
 */
#define GDA_REGISTER_LOGGING gda_register_logging

/**
 * @brief Macro for gda_register_thread_to_context function.
 */
#define GDA_REGISTER_THREAD_TO_CONTEXT gda_register_thread_to_context

/**
 * @brief Macro for gda_unregister_logging function.
 */
#define GDA_UNREGISTER_LOGGING gda_unregister_logging

/**
 * @brief GDA error codes.
 */
typedef enum GDA_ER
{
    GDA_OK = 0,                       /**< No error*/
    GDA_ER_DLT_CTXID_NOT_SET,         /**< DLT Context ID not set*/
    GDA_ER_CTX_MEM_NOT_ALLOCATED,     /**< Memory not allocated for GDA context*/
    GDA_ER_NO_MEM,                    /**< Could not allocate memory*/
    GDA_ER_NO_CTX_AVAILABLE,          /**< No registered contexts available*/
    GDA_ER_CTX_THREADS_LIMIT_REACHED, /**< Max no. of threads that can be registered to context reached*/
    GDA_ER_CTX_NOT_REGISTERED,        /**< No such context available, context may not be registered*/
    GDA_ER_THREAD_ID_IS_NULL          /**< ThreadID is NULL, Not registered to context*/
}GDA_ER;

/**
 * @brief Structure that holds the DLT related info for each context.
 * This Structure is initialized and filled by GDA.
 */
typedef struct GDA_CTX
{
    DltContext      dlt_ctx;        /**< DLT logging context that application registers with*/
    DltLogLevelType dlt_log_level;  /**< Application context log level */
    uint32_t        dlt_service_id; /**< DLT serviceID of the injection message */
}GDA_CTX;

/**
 * @brief Register with GDA
 *
 * Initializes the GDA Context, GDA registers a logging function with GStreamer. GDA also registers
 * the application context with DLT and sets the log level of the context to be
 * the max value in GST_DEBUG env variable. It also registers for log level changed callback
 * and injection callback with DLT. p_gda_ctx is initialized and filled by GDA.
 *
 * @param[out] p_gda_ctx                  Provides GDA context to be initialized
 * @param[in]  dlt_ctx_id                 Provides DLT contextID for DltContext registration
 * @param[in]  dlt_ctx_desc               Provides DLT context description
 * @param[in]  injection_service_id       Provides DLT serviceID of injection callback
 * @return @ref GDA_OK                    Registration successful
 * @return @ref GDA_ER_NO_MEM             Could not allocate memory
 * @return @ref GDA_ER_DLT_CTXID_NOT_SET  DLT Context ID not set
 * @return @ref GDA_ER_CTX_MEM_NOT_ALLOCATED Memory not allocated for GDA nontext
 */
GDA_ER gda_register_logging (GDA_CTX * p_gda_ctx, const char * dlt_ctx_id,
                           const char * dlt_ctx_desc, uint32_t injection_service_id);

/**
 * @brief Registers threadID to GDAContext.
 *
 * @param[in] thread_id                   ThreadID of GStreamer threads.
 * @param[in] dlt_ctx_id                  DLT Context ID
 * @return @ref GDA_OK                    Registration successful
 * @return @ref GDA_ER_NO_MEM             Could not allocate memory
 * @return @ref GDA_ER_NO_CTX_AVAILABLE   No registered contexts available
 * @return @ref GDA_ER_THREAD_ID_IS_NULL  ThreadID is NULL, Not registered to context
 * @return @ref GDA_ER_CTX_NOT_REGISTERED No such context available, context may not be registered
 * @return @ref GDA_ER_CTX_THREADS_LIMIT_REACHED Max no. of threads that can be registered to context reached
 */
GDA_ER gda_register_thread_to_context (GThread * thread_id, const char * dlt_ctx_id);

/**
 * @brief Unregister with GDA.
 *
 * Unregisters the GDA_CTX and the DltContext. Application needs to unregister each of its pipeline with GDA
 * @param[in] p_gda_ctx                   Pointer to GDA Context to be unregistered.
 * @return @ref GDA_OK                    Registration successful
 * @return @ref GDA_ER_NO_CTX_AVAILABLE   No registered contexts available
 * @return @ref GDA_ER_CTX_NOT_REGISTERED No such context available, context may not be registered
 */
GDA_ER gda_unregister_logging (GDA_CTX * p_gda_ctx);

#ifdef __cplusplus
}
#endif

/*! @file gst_dlt_adapter.h
 *
 * \mainpage Purpose
 *
 *  The standard mechanism for GStreamer log generation traces the logs to standard error.
 *  To get these logs and also to change the logging level, developer has to connect
 *  to the target via serial cable or network as a root user. Furthermore, the generated logs
 *  are lost because project disables the journal which stores these log messages generated over
 *  standard error. The main goal of this work package is to trace these logs via DLT.\n
 *  GDA makes this easy by providing interfaces to route the GStreamer logs to DLT.
 *  GDA is a helper library to route the GStreamer logs, however
 *  application can also implement it by its own.\n
 *
 * \page UC Use Cases
 *  The main usecase of GDA is to the trace the GStreamer logs via DLT.
 *  The figure below shows the usecase of GDA.
 *  \startuml UseCases
 *  left to right direction
 *  skinparam packageStyle rect
 *  scale 4/5
 *  actor DLTViewerUser
 *  rectangle Target {
 *  [DLT]
 *  actor GSTApplication
 *  GSTApplication -[hidden]- [DLTViewerUser]
 *      rectangle GDA {
 *          [GSTApplication] -- (Logging)
 *          (Logging) -- [DLT]
 *          (Logging) ..> (Routing) : <<includes>>
 *          (Logging) ..> (Filtering) : <<includes>>
 *          (Injecting filter configuration) ..> (Filtering) : <<extends>>
 *          [DLTViewerUser] ..> (Injecting filter configuration)
 *          }
 *    }
 *  \enduml
 *
 * \section Logging Use Case 2.1: Logging GStreamer logs
 * <table>
 * <tr><th>Use Case          <th>Logging
 * <tr><td>Responsibilities</td><td>GStreamer Application <br>- Registers the application with DLT.
 * <br>- Initializes the GStreamer. <br>- Register with GDA to route the GStreamer logs via DLT.
 * <br><br>GDA <br>- Generates the GStreamer logs <br>- Filters the logs.
 * <br>- Routes the GStreamer logs to DLT.</td>
 * <tr><td>Trigger</td><td>Application wants to route GStreamer logs to DLT.</td>
 * <tr><td>Pre-Condition</td><td>DLT daemon is running.</td>
 * <tr><td>Result</td><td>GStreamer logs will be routed to DLT.</td>
 * <tr><td>Exceptions</td><td>GDA registration fails.</td>\n
 * </table>
 *
 * \section FILTERLOGS Use Case 2.2: Filtering GStreamer logs by Injecting filter configuration
 *  GDA also supports filtering of logs at runtime. To enable the logs only for a certain debug category
 *  at a certain level, DLTViewerUser needs to send an injection message via DLT-Viewer. Refer to the
 *  \ref UserManual to know more.
 *
 * \page RQ Requirements
 *
 * <table>
 * <tr><th>Sl.No          <th>Functional Requirements
 * <tr><td>1</td><td>GDA shall route the GStreamer logs to DLT.</td>
 * <tr><td>2</td><td>GDA shall allow the GStreamer log level to be changed at run time via DLT-Viewer or Trace TTFis.</td>
 * </table>
 *
 * <table>
 * <tr><th>Sl.No          <th>Characteristics
 * <tr><td>1</td><td>GDA shall operate with the initial log level to be the value set in GST_DEBUG environmant variable.</td>
 * <tr><td>2</td><td>GDA shall filter the logs based on the user input <category:level> for injection message sent via DLT-Viewer.</td>
 * <tr><td>3</td><td>GDA shall support wildcard as an input for category to filter the logs.</td>
 * <tr><td>4</td><td>GDA shall support single as well as multiple pipelines.</td>
 * <tr><td>5</td><td>GDA shall be compatible with GStreamer-0.10 as well as GStreamer-1.0.</td>
 * </table>
 *
 * \page SV Structural View
 *  For the ease of debugging GDA provides interfaces that can be used to trace the GStreamer logs
 *  to DLT. Currently the GStreamer log level needs to be set via command line
 *  by exporting GST_DEBUG environment variable, GDA provides an alternative.\n
 *  The figure below shows the components involved in routing the GStreamer logs to DLT
 *  and their interaction.
 *
 *   @startuml StructuralView
 *    skinparam packageStyle rectangle
 *    rectangle StructuralView {
 *     component GDA #ffff00
 *          [Application] ..> [DLT]
 *          [Application] ..> GDA
 *          [Application] ..> [GST]
 *          GDA ..> [DLT]
 *          GDA ..> [GST]
 *          [GST] ..> GDA
 *          }
 *   @enduml
 *
 *  Application has to register itself with DLT and initialize GStreamer. Then it has to register with GDA with a context.
 *  GDA registers this application context with DLT and the generated GStreamer logs
 *  will be traced in this dlt context. GDA also registers a logging function callback with GStreamer.
 *  This GDA callback is called whenever a macro like GST_DEBUG or similar is used in GStreamer.
 *  It is in this callback GDA generates, filters and routes the GStreamer logs to DLT.\n
 *  GDA also provides the feasibility for the user to change the logging level and filter the
 *  GStreamer logs at runtime. GDA is compatible with GStreamer-0.10 as well as GStreamer-1.0.
 *  GDA supports single as well as multiple pipelines.
 * \note Refer to @ref UserManual for more information on changing of the logging level and filtering of the
 *  GStreamer logs at runtime.
 *
 * \page DV Dynamic View
 *  GDA functionality of rerouting the GStreamer logs can split into Registration,
 *  Generating-Filtering-Routing GStreamer logs and Unregistration.
 *
 *  @startuml DynamicView
 * skinparam SequenceMessageAlign center
 *  scale 1/2
 *  actor Application
 *  participant GDA #ffff00
 *  participant GST
 *  participant DLT
 *  ref over Application, GDA , GST, DLT : Registration
 *  ref over Application, GDA , GST, DLT : Generating, Filtering and Routing GST logs
 *  ref over Application, GDA , GST, DLT : Unregistration
 * @enduml
 *
 * Below figures give the insights into each of these steps.
 *
 * \note Refer to @ref ApplicationDeveloperManual for sample code snippets.
 *
 * \section DV1 Registration
 *  Application that needs to route the GStreamer logs to DLT needs to first
 *  register the application with DLT using DLT_REGISTER_APP(). Then it needs to initialize
 *  GStreamer with gst_init(). Only after the GStreamer is initialized, application
 *  needs to register itself with GDA using @ref gda_register_logging().\n
 *  During registration, GDA registers a logging function with GStreamer which is called
 *  whenever a logging macro is called in GStreamer. GDA also registers the application
 *  context with DLT. The GStreamer logs from this pipeline will be traced in this context.
 *  GDA parses the GST_DEBUG environment variable to get the max level and sets that
 *  level to DLT during context registration.\n
 *  To support changing of log level at runtime GDA registers for a callback with DLT that is
 *  called when the log level for the application context is changed. In this callback GDA
 *  sets that log level to GStreamer. To support filtering of logs GDA also registers for
 *  injection callback with DLT.
 *
 * \startuml GDARegistration
 * skinparam SequenceMessageAlign center
 * scale 4/5
 *  actor Application
 *  participant GDA #ffff00
 *  participant GST
 *  participant DLT
 *  Application -> DLT: DLT_REGISTER_APP()
 *  DLT --> Application
 *  Application -> GST: gst_init()
 *  GST --> Application
 *  Application -> GDA: GDA_REGISTER_LOGGING()
 *  GDA -> GST: gst_debug_add_log_function()
 *  GST --> GDA
 *  GDA -> GDA : Parse GST_DEBUG to get the max level
 *  GDA -> DLT: DLT_REGISTER_CONTEXT_LL_TS
 *  note left
 *      Register the context
 *      and set the log level
 *  end note
 *  DLT --> GDA
 *  GDA -> DLT: DLT_REGISTER_LOG_LEVEL_CHANGED_CALLBACK
 *  DLT --> GDA
 *  GDA -> DLT: DLT_REGISTER_INJECTION_CALLBACK_WITH_ID
 *  DLT --> GDA
 *  GDA --> Application: :GDA_ER
 * \enduml
 *
 * \section DV2 Generating, Filtering and Routing GStreamer logs
 *  After registration application can then create its pipeline and start the playback. Whenever a
 *  macro like GST_DEBUG or similar is used in GStreamer, the registered GDA callback
 *  is called. It is in this callback GDA generates, filters and routes the GStreamer logs to DLT.
 *
 * \startuml GDAGenerationOfGSTLogs
 * skinparam SequenceMessageAlign center
 * scale 4/5
 *  actor Application
 *  participant GDA #ffff00
 *  participant GST
 *  participant DLT
 *  Application -> GST: CreatePipelineAndStartPlayback()
 *  loop playback
 *  alt GST_DEBUG or similar macro ?
 *  GST -> GDA: LoggingFuncCallBack()
 *  GDA --> GDA : GenerateLog
 *  alt Injection message was sent via DLT?
 *  GDA --> GDA : FilterLog
 *  end
 *  GDA -> DLT: Route GST logs to DLT
 *  DLT --> GDA
 *  GDA --> GST
 *  end
 *  end
 * \enduml
 *
 * \note Refer to @ref UserManual for more information on filtering of the GStreamer logs at runtime.
 *
 * \section DV3 Unregistration
 *  Application can unregister with GDA using @ref gda_unregister_logging(). During unregistration,
 *  GDA unregisters the application context with DLT and removes the logging function.
 *
 * \startuml GDAUnregistration
 * skinparam SequenceMessageAlign center
 * scale 4/5
 *  actor Application
 *  participant GDA #ffff00
 *  participant GST
 *  participant DLT
 *  Application -> GDA: GDA_UNREGISTER_LOGGING()
 *  GDA -> GST : gst_debug_remove_log_function()
 *  GST --> GDA
 *  GDA -> DLT : DLT_UNREGISTER_CONTEXT()
 *  DLT --> GDA
 *  GDA --> Application: :GDA_ER
 * \enduml
 *
 * \section MP Multiple Pipelines
 *  In multiple pipeline use case GDA logs the traces from each pipeline in a different
 *  DLT context. So the logs from the pipelines can be easily differentiated.
 *  Registration, Generating-Filtering-Routing GStreamer logs and Unregistration in case of multiple
 *  pipelines are explained below.
 *
 * \note
 *  - Refer to @ref ApplicationDeveloperManual for sample code snippets.
 *  - It is not necessary for the pipelines to share the g_main_context in order to use GDA.
 *
 * \subsection DV4 Registration
 *  Application needs to register each of its pipeline with GDA with a
 *  different DLT context and GDA routes the logs from each pipeline in their respective context.
 *  GDA differentiates the logs from each pipeline based on the threadID,
 *  hence the application needs to register the GStreamer threads using gda_register_thread_to_context().\n
 *  GStreamer threadIDs can be obtained by registering the sync handler on the bus.
 *  Every time a message of type GST_STREAM_STATUS_TYPE_CREATE or GST_STREAM_STATUS_TYPE_ENTER
 *  is posted on the bus, application needs to get the threadID with g_thread_self() and register
 *  the threadID with GDA.
 * \startuml MultiplepipelinesRegistration
 * skinparam SequenceMessageAlign center
 * scale 4/5
 *  actor Application
 *  participant Pipeline1
 *  participant Pipeline2
 *  participant GDA #ffff00
 *  participant GST
 *  note right: Application has registered and has initialized GST
 *  == Register Logging ==
 *  Application -> GDA: GDA_REGISTER_LOGGING(CTX1)
 *  create Pipeline1
 *  Application ->Pipeline1
 *  Pipeline1 -> GST: CreatePipeline()
 *  Pipeline1 -> GST: gst_bus_set_sync_handler()
 *  Pipeline1 -> GST: StartPlayback()
 *  Application -> GDA: GDA_REGISTER_LOGGING(CTX2)
 *  create Pipeline2
 *  Application ->Pipeline2
 *  Pipeline2 -> GST: CreatePipeline()
 *  Pipeline2 -> GST: gst_bus_set_sync_handler()
 *  Pipeline2 -> GST: StartPlayback()
 *  == Register GST Threads ==
 *  alt GST_STREAM_STATUS_TYPE_CREATE or \nGST_STREAM_STATUS_TYPE_ENTER posted on bus
 *  Pipeline1 -> GDA : GDA_REGISTER_THREAD_TO_CONTEXT(g_thread_self(), CXT1);
 *  end
 *  alt GST_STREAM_STATUS_TYPE_CREATE or \nGST_STREAM_STATUS_TYPE_ENTER posted on bus
 *  Pipeline2 -> GDA : GDA_REGISTER_THREAD_TO_CONTEXT(g_thread_self(), CXT2);
 *  end
 * \enduml
 *
 * \subsection DV5 Generating, Filtering and Routing GST logs
 *  When a macro like GST_DEBUG or similar is used in GStreamer the logging function is
 *  called from GStreamer. Since the logging function is called in the context of the
 *  GStreamer thread, GDA gets the threadID using g_thread_self(). Based on this threadID
 *  GDA routes the log from that thread to the context with which it is registered in GDA.\n
 *  Since the logs are differentiated based on the threadIDs and some threads are
 *  shared between the pipelines, logs from such threads and the gst_init are traced
 *  in the GDA default context.
 *
 * \startuml MultiplepipelinesGenerationOfGSTLogs
 * skinparam SequenceMessageAlign center
 * scale 4/5
 *  participant GST
 *  participant GDA #ffff00
 *  participant DLT
 *  loop playback
 *  alt GST_DEBUG or similar macro ?
 *  GST -> GDA: LoggingFuncCallBack()
 *  GDA --> GDA : GenerateLog
 *  alt Injection message sent via DLT?
 *  GDA --> GDA : FilterLog
 *  end
 *  alt thread belongs to Pipeline1
 *      GDA -> DLT: Route GST logs to DLT with CTX1
 *  else thread belongs to Pipeline2
 *      GDA -> DLT: Route GST logs to DLT with CTX2
 *  else
 *      GDA -> DLT: Route GST logs to DLT with GDAC
 *  end
 *  DLT --> GDA
 *  GDA --> GST
 *  end
 *  end
 * \enduml
 *
 * \subsection DV6 Unregistration
 *  Each pipeline can  unregister with GDA using @ref gda_unregister_logging(). During
 *  unregistration, GDA unregisters the context with DLT. The logging function callback
 *  is unregistered only during the last GDA unregistration.
 *
 * \startuml MultiplepipelinesUnRegistration
 * skinparam SequenceMessageAlign center
 * scale 4/5
 *  actor Application
 *  participant Pipeline1
 *  participant Pipeline2
 *  participant GST
 *  participant GDA #ffff00
 *  participant DLT
 *  Application -> GDA: GDA_UNREGISTER_LOGGING(CTX1)
 *  GDA -> DLT : DLT_UNREGISTER_CONTEXT()
 *  DLT --> GDA
 *  GDA --> Application: :GDA_ER
 *  Application -> GDA: GDA_UNREGISTER_LOGGING(CTX2)
 *  GDA -> GST : gst_debug_remove_log_function()
 *  GST --> GDA
 *  GDA -> DLT : DLT_UNREGISTER_CONTEXT()
 *  DLT --> GDA
 *  GDA --> Application: :GDA_ER
 * \enduml
 *
 * \page ApplicationDeveloperManual
 * \section REGISTER_UNREGISTER GDA register and unregister
 * @code
 *  int main (int argc, char *argv[])
 *  {
 *       GDA_ER err = GDA_OK;
 *       GDA_CTX gda_ctx;
 *       DLT_REGISTER_APP ("GDLT", "GST-DLT Application")
 *       gst_init (&argc, &argv);
 *       err = GDA_REGISTER_LOGGING (&gda_ctx, "GDCT","GStreamer context", DLT_SERVICEID);
 *       CreatePipelineStartPlayback();
 *       err = GDA_UNREGISTER_LOGGING (gda_ctx);
 *  }
 * @endcode
 *
 * \section GST_THREAD_ID Get the threadID of GStreamer threads
 * @code
 *  static gboolean gst_watch (GstBus * bus, GstMessage * message, gpointer data)
 *  {
 *      GstStreamStatusType type;
 *      sync_handler_data_t *handler_data = (sync_handler_data_t*) data;
 *      if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_EOS)
 *      {
 *          g_main_loop_quit (handler_data->loop);
 *      }
 *      if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_STREAM_STATUS)
 *      {
 *          gst_message_parse_stream_status (message, &type, NULL);
 *          if (type == GST_STREAM_STATUS_TYPE_CREATE || type == GST_STREAM_STATUS_TYPE_ENTER)
 *          {
 *              GDA_REGISTER_THREAD_TO_CONTEXT (g_thread_self(), (const char* )handler_data->dlt_contextID);
 *          }
 *      }
 *  }
 *
 *  static void CreatePipelinesAndStartPlayback(void)
 *  {
 *   .
 *   .
 *  #ifdef GST_VERSION_1
 *      gst_bus_set_sync_handler (bus, (GstBusSyncHandler)gst_watch, &sync_handler_data, NULL);
 *  #else
 *      gst_bus_set_sync_handler (bus, (GstBusSyncHandler)gst_watch, &sync_handler_data);
 *  #endif
 *   .
 *   .
 *  }
 * @endcode
 *
 * \page UserManual
 * \section CLL Change logging Level
 *  To change the logging level  user has to double click on the GDA registered application context
 *  and change the level in the Log Level tab as below.
 *
 *  @image html GDA_Change_Level.png
 *  @image latex GDA_Change_Level.png
 *
 *  \note Changing the default log level of DLT will not affect the GDA, only
 *  when the log level of the GDA registered application context is changed GDA gets a log level changed callback from DLT.
 *  \n
 *  \section FL Filtering of logs
 *  User can enable the logs only for a certain debug category at a certain level,
 *  by sending an injection message via DLT-Viewer.
 *  It is important that Application ID, Context ID and Service ID match the GStreamer
 *  Application ID that the application has registered with DLT and
 *  the Context ID and Service ID that the application has registered with GDA.\n
 *  Format of data field : <debug_category:level>. Wildcard is supported as well.
 *  DLT log level will be set as default level for the debug category if level is not specified.\n
 *  e.g *filesrc*:3,wavparse:5;alsa*\n
 *  This feature works for multiple pipelines as well. An injection message with the required debug category
 *  can be sent for each of pipelines with the corresponding Context ID and Service ID that
 *  the application has registered them with GDA.
 *
 *  @image html GDA_inj.png
 *  @image latex GDA_inj.png
 *\n
 *  @image html GDA_inj_wildcard.png
 *  @image latex GDA_inj_wildcard.png
 *
 */


#endif /* __log_gst_dlt_H__ */
